// $Id: CExpGainEditor.cpp,v 1.3 2007/03/03 14:46:07 paul Exp $

/*
 * All contents of this source code are copyright 2007 Exp Digital Uk.
 * This file forms part of the example projects for the Infinity API.
 * You may not redistribute either the source code or the compiled binary.
 * The Infinity API is covered by a fair use licence which you should have recieved with this
 * file. If you didnt receieve the licence, please contact us via http://www.expdigital.co.uk
 */

#include "CExpGain.hpp"
#include "CExpGainEditor.hpp"
using Exponent::ExpGain::CExpGain;
using Exponent::ExpGain::CExpGainEditor;

//	===========================================================================
EXPONENT_CLASS_IMPLEMENTATION(CExpGainEditor, CVstWindow);

//	===========================================================================
CExpGainEditor::CExpGainEditor(AudioEffect *theEffect, const CRect &windowArea, CWindowAttributes *attributes) 
			  : CVstWindow(theEffect, windowArea, attributes)
{
	EXPONENT_CLASS_CONSTRUCTION(CExpGainEditor);

	/* 
	 * Store the default value of the slider and a text representation
	 */
	const double value = theEffect->getParameter(0);
	CString theValue;
	theValue.setStringWithFormat("Gain %03li%%", (long)(value * 100.0));

	/*
	 * The text label and slider we are storing in an automatic pointer. This works just like a normal pointer, but it will automatically destroy itself!
	 * You dont need to destroy an automatic pointer. Just create it and then forget about it :)
	 */
	m_label  = new CTextLabel(this->getControlRoot(), CControl::CCONTROL_NO_ID_REQUIRED, CRect(0, 0, windowArea.getWidth(), 20), theValue);
	m_slider = new CSlider(this->getControlRoot(), CControl::CCONTROL_NO_ID_REQUIRED, CRect(50, 30, 20, 100), this);

	// Set the value for the slider (inverted to suit directionality of a gain slider)
	m_slider->setValue(1.0 - value);

	/*
	 * We have to get access to the pointer (and notice it has to be mutable because the reference count is incremented)
	 * This call adds the control to the window
	 */
	this->getControlRoot()->addControl(m_label.getMutablePointer());
	this->getControlRoot()->addControl(m_slider.getMutablePointer());

	// We want to center the text
	m_label->setTextFormat(CGraphics::e_centerCenterAlign);

	// If you want to set a custom background, do it something like this
	//this->getControlRoot()->setBackgroundImage(new CPNGImage("/path/to/myimage/back.png"));
}

//	===========================================================================
CExpGainEditor::~CExpGainEditor()
{
	EXPONENT_CLASS_DESTRUCTION(CExpGainEditor);
}

//	===========================================================================
void CExpGainEditor::handleActionEvent(const CActionEvent &event)
{
	/*
	 * We recieve this notification when a control has changed (ie the user has dragged on the fader)
	 * This means that the value of the slider has changed, and we need to update the effect
	 */
	if (effect)
	{
		/*
		 * We call this that the effect knows that we are the ones giving it an update, not the host, and therefore
		 * it doesnt need to notify us about the change, as we have done it!
		 */
		m_setFromEditor = true;

		/*
		 * First lets notify the effect,
		 * we do this using setParameterAutomated, which also notifies the host so they can write parameter automation
		 */
		effect->setParameterAutomated(0, (float)event.getControl()->getValue());

		/*
		 * Now we have finished notifying the plugin, we want to turn back on the notifications
		 */
		m_setFromEditor = false;

		/*
		 * Now we update the text display to display something relevant to the level we have
		 * Notice that we get the value from the control. We could just as easily call down to the plugin to get the value
		 */
		CString text;
		text.setStringWithFormat("Gain %03li%%", (long)(event.getControl()->getValue() * 100.0));
		m_label->setText(text);
	}
}

//	===========================================================================
void CExpGainEditor::setParameter(const long index, const double value)
{
	/*
	 * We recieve this when the parameter has been updated from outside of the editor
	 */
	m_slider->setValue(1.0 - value);
	m_slider->update();

	/*
	 * We also need to update the text,
	 * here again, we could just as easily call the effects getParameter call
	 */
	CString text;
	text.setStringWithFormat("Gain %03li%%", (long)(value * 100.0));
	m_label->setText(text);
}